/**
 * \file sdc_session.c
 *
 * \brief Functions of session interface
 *
 * Redirect all session specific functions to the architecture dependent version
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <string.h>
#include <sdc_session.h>
#include <sdc_perm.h>
#include <private/sdc_arch.h>

/* Definitions types and defaults */

/* Functions */

/* initialize architecture independent parts */
static sdc_error_t sdc_session_init(sdc_session_t *session)
{
    memset(session->unaligned_buffer, 0, SDC_ARCH_MAX_UNALIGNED_LEN);
    session->unaligned_buffer_filllevel = 0;
    session->inlen_cnt = 0;
    session->sequence.crypto_op = SDC_OP_NO_CRYPTO;
    session->sequence.op_state = SDC_NO_OP;
    session->iuf_wrap_unwrap.total_inlen = 0;
    session->iuf_wrap_unwrap.total_aadlen = 0;
    session->iuf_wrap_unwrap.tag_len = 0;

    return SDC_OK;
}


/* deinitialize architecture independent parts */
static sdc_error_t sdc_session_deinit(sdc_session_t *session)
{
    sdc_intern_clear_session_confidential(session);

    return SDC_OK;
}

sdc_error_t sdc_open_session(sdc_session_t **session)
{
    sdc_error_t err;

    if (!session)
        return SDC_SESSION_INVALID;

    *session = malloc(sizeof(sdc_session_t));
    if ((*session) == NULL)
        return SDC_NO_MEM;

    err = sdc_session_init(*session);
    if (err == SDC_OK) {
        err = sdc_arch_session_init(*session);

        if (err != SDC_OK)
            sdc_session_deinit(*session);
    }

    if (err != SDC_OK) {
        free(*session);
        *session = NULL;
    }

    return err;
}

sdc_error_t sdc_close_session(sdc_session_t *session)
{
    sdc_error_t arch_err;
    sdc_error_t err;

    if (!session)
        return SDC_SESSION_INVALID;

    arch_err = sdc_arch_session_deinit(session);
    err = sdc_session_deinit(session);

    /* always free */
    free(session);

    if (err == SDC_OK)
        err = arch_err;

    return err;
}

sdc_error_t sdc_session_load_storage_key(sdc_session_t *session, sdc_key_id_t kid)
{
    if (!session)
        return SDC_SESSION_INVALID;

    if (kid == SDC_FLAG_INVALID_KID) {
        return SDC_KEY_INVALID;
    }

    return sdc_arch_session_load_storage_key(session, kid);
}

sdc_error_t sdc_session_load_builtin_key(sdc_session_t *session,
                                         const sdc_key_fmt_t fmt,
                                         const sdc_key_len_t len,
                                         const uint8_t *modifier,
                                         const size_t modifier_len,
                                         const bool secret_modifier,
                                         const sdc_permissions_t *perms)
{
    sdc_error_t err;
    sdc_key_common_t key_config;
    sdc_key_modifier_t mod = {
            .data = (uint8_t *)modifier,
            .len = modifier_len,
            .is_secret = secret_modifier,
    };

    if (!session)
        return SDC_SESSION_INVALID;

    if ((!modifier && modifier_len) || !perms)
        return SDC_INVALID_PARAMETER;

    if ((len < SDC_KEY_LEN_FIRST) || (len >= SDC_KEY_LEN_END))
        err = SDC_KEYLEN_INVALID;

    if ((fmt < SDC_KEY_FMT_FIRST) || (fmt >= SDC_KEY_FMT_END))
        err = SDC_KEY_FMT_INVALID;

    err = sdc_intern_check_permission_range(perms);
    if (err != SDC_OK)
        return err;

    key_config.fmt = fmt;
    key_config.len = len;
    key_config.perms = perms;

    return sdc_arch_session_load_builtin_key(session,
                                             &key_config,
                                             &mod);
}

sdc_error_t sdc_import_formatted_autoload_key_with_secret_mod(
    sdc_session_t *session,
    const uint8_t *formatted_data, const size_t formatted_len,
    const uint8_t *secret_mod_data, const size_t secret_mod_len)
{
    return sdc_intern_formatted_autoload_key_with_secret_mod(
               session,
               formatted_data, formatted_len,
               secret_mod_data, secret_mod_len);
}

sdc_error_t sdc_session_unset_key(sdc_session_t *session)
{
    if (!session)
        return SDC_SESSION_INVALID;

    return sdc_arch_session_unset_key(session);
}
